#include <p30fxxxx.h>
#include "timer.h"

/********************************************************************
*    Function Name:  CloseTimer1                                    *
*    Description:    This routine disables the Timer1 and its       *
*                    interrupt enable and flag bits.                *
*    Parameters:     None                                           *
*    Return Value:   None                                           *
********************************************************************/
void CloseTimer1(void)
{
    IEC0bits.T1IE = 0;      /* Disable the Timer1 interrupt */
    T1CONbits.TON = 0;      /* Disable timer1 */
    IFS0bits.T1IF = 0;      /* Clear Timer interrupt flag */
}

/********************************************************************
*    Function Name:  CloseTimer2                                    *
*    Description:    This routine disables the Timer2 and its       *
*                    interrupt and flag bits.                       *
*    Parameters:     None                                           *
*    Return Value:   None                                           *
********************************************************************/
void CloseTimer2(void)
{
    IEC0bits.T2IE = 0;      /* Disable the Timer2 interrupt */
    T2CONbits.TON = 0;      /* Disable timer2 */
    IFS0bits.T2IF = 0;      /* Clear Timer interrupt flag */
}
/********************************************************************
*    Function Name:  CloseTimer3                                    *
*    Description:    This routine disables the Timer3 and its       *
*                    interrupt and flag bits.                       *
*    Parameters:     None                                           *
*    Return Value:   None                                           *
********************************************************************/
void CloseTimer3(void)
{
    IEC0bits.T3IE = 0;      /* Disable the Timer3 interrupt */
    T3CONbits.TON = 0;      /* Disable timer3 */
    IFS0bits.T3IF = 0;      /* Clear Timer interrupt flag */
}

/********************************************************************
*    Function Name:  CloseTimer23                                   *
*    Description:    This routine disables the Timer2 and 3 and its *
*                    interrupt and flag bits.                       *
*    Parameters:     None                                           *
*    Return Value:   None                                           *
********************************************************************/
void CloseTimer23(void)
{
    IEC0bits.T3IE = 0;      /* Disable the Timer3 interrupt */
    if (T2CONbits.T32 == 1)
        T2CONbits.TON = 0;  /* Disable Timer2 */
    IFS0bits.T3IF = 0;      /* Clear Timer3 interrupt flag */
}

/*******************************************************************
*    Function Name: ConfigIntTimer1                                *
*    Description:   This Function Configures Interrupt and sets    *
*                   Interrupt Priority                             *
*    Return Value:  None                                           *
*******************************************************************/
void ConfigIntTimer1(unsigned int config)
{
    IFS0bits.T1IF = 0;                   /* clear IF bit */
    IPC0bits.T1IP = (config &0x0007);    /* assigning Interrupt Priority */
    IEC0bits.T1IE = (config &0x0008)>>3; /* Interrupt Enable /Disable */
}

/*******************************************************************
*    Function Name: ConfigIntTimer2                                *
*    Description:   This Function Configures Interrupt and sets    *
*                   Interrupt Priority                             *
*    Parameters:    unsigned int config                            *
*    Return Value:  None                                           *
*******************************************************************/
void ConfigIntTimer2(unsigned int config)
{
    IFS0bits.T2IF = 0;                   /* clear IF bit */
    IPC1bits.T2IP = (config &0x0007);    /* assigning Interrupt Priority */
    IEC0bits.T2IE = (config &0x0008)>>3; /* Interrupt Enable /Disable */
}

/*******************************************************************
*    Function Name: ConfigIntTimer3                                *
*    Description:   This Function Configures Interrupt and sets    *
*                   Interrupt Priority                             *
*    Parameters:    unsigned int config                            *
*    Return Value:  None                                           *
*******************************************************************/
void ConfigIntTimer3(unsigned int config)
{
    IFS0bits.T3IF = 0;                   /* clear IF bit */
    IPC1bits.T3IP = (config &0x0007);    /* assigning Interrupt Priority */
    IEC0bits.T3IE = (config &0x0008)>>3; /* Interrupt Enable /Disable */
}

/*******************************************************************
*    Function Name: ConfigIntTimer23                               *
*    Description:   This Function Configures Interrupt and sets    *
*                   Interrupt Priority                             *
*    Parameters:    unsigned int config                            *
*    Return Value:  None                                           *
*******************************************************************/
void ConfigIntTimer23(unsigned int config)
{
    IFS0bits.T3IF = 0;                   /* clear IF bit */
    IPC1bits.T3IP = (config &0x0007);    /* assigning Interrupt Priority */
    IEC0bits.T3IE = (config &0x0008)>>3; /* Interrupt Enable /Disable */
}

/********************************************************************
*    Function Name:  OpenTimer1                                     *
*    Description:    This routine configures the timer control regis*
*                    ter and timer period register.                 *
*    Parameters:     config: bit definitions to configure Timer1    *
*                    period: Value to be loaded to PR reg           *
*    Return Value:   None                                           *
********************************************************************/
void OpenTimer1(unsigned int config,unsigned int period)
{
    TMR1  = 0;          /* Reset Timer1 to 0x0000 */
    PR1   = period;     /* assigning Period to Timer period register */
    T1CON = config;     /* Configure timer control reg */
}

/********************************************************************
*    Function Name:  OpenTimer2                                     *
*    Description:    This routine configures the timer control regis*
*                    ter and timer period register.                 *
*    Parameters:     config: bit definitions to configure Timer2    *
*                    period: value to be loaded to PR reg           *
*    Return Value:   None                                           *
********************************************************************/
void OpenTimer2(unsigned int config,unsigned int period)
{
    TMR2  = 0;          /* Reset Timer2 to 0x0000 */
    PR2   = period;     /* assigning Period to Timer period register */
    T2CON = config;     /* configure control reg */
    T2CONbits.T32 = 0;
}

/********************************************************************
*    Function Name:  OpenTimer3                                     *
*    Description:    This routine configures the timer control regis*
*                    ter and timer period register.                 *
*    Parameters:     config: bit definitions to configure Timer3    *
*                    period: value to be loaded to PR reg           *
*    Return Value:   None                                           *
********************************************************************/
void OpenTimer3(unsigned int  config,unsigned int period)
{
    TMR3  = 0;      /* Reset Timer3 to 0x0000 */
    PR3   = period; /* assigning Period to Timer period register.*/
    T3CON = config; /* configure timer control reg */
}

/********************************************************************
*    Function Name:  OpenTimer23                                    *
*    Description:    This routine configures the timer control regis*
*                    ter and timer period register.                 *
*    Parameters:     config: bit definitions to configure Timer2    *
*                    period: value to be loaded to PR reg           *
*    Return Value:   None                                           *
********************************************************************/
void OpenTimer23(unsigned int  config,unsigned long period)
{
    TMR2 = 0;               /* Reset Timer2 to 0x0000 */
    TMR3 = 0;               /* Reset Timer3 to 0x0000 */
    PR2 = period;           /* assigning Period to PR2 */
    PR3 = period>>16;       /* Period to  PR3 Register */
    T2CON =  config;        /* Configure timer control reg */
    T2CONbits.T32 = 1;
}

/********************************************************************
*    Function Name:  ReadTimer1                                     *
*    Description:    This routine reads the 16-bit value from       *
*                    Timer Register.                                *
*    Parameters:     None                                           *
*    Return Value:   unsigned int: Timer  16-bit value              *
********************************************************************/
unsigned int ReadTimer1(void)
{
    return (TMR1);      /* Return the Timer1 register */
}

/********************************************************************
*    Function Name:  ReadTimer2                                     *
*    Description:    This routine reads the 16-bit value from       *
*                    Timer2 Register.                               *
*    Parameters:     None                                           *
*    Return Value:   unsigned int: Timer  16-bit value              *
********************************************************************/
unsigned int ReadTimer2(void)
{
    return (TMR2);      /* Return the Timer2 register */
}

/********************************************************************
*    Function Name:  ReadTimer3                                     *
*    Description:    This routine reads the 16-bit value from       *
*                    Timer3 Register.                               *
*    Parameters:     None                                           *
*    Return Value:   unsigned int: Timer  16-bit value              *
********************************************************************/
unsigned int ReadTimer3(void)
{
    return (TMR3);      /* Return the Timer3 register */
}

/********************************************************************
*    Function Name:  ReadTimer23                                    *
*    Description:    This routine reads the 32-bit value from       *
*                    Timer3 and Timer2.                             *
*    Parameters:     None                                           *
*    Return Value:   unsigned long: Timer2 and Timer3  32-bit value *
********************************************************************/
unsigned long ReadTimer23(void)
{
    unsigned long timer;
    timer = TMR2;                        /* Copy Timer2 into timer low 16bit */
    timer |= (unsigned long)TMR3HLD<<16; /* shift  16 time as TMR 3 
                                            contains MSB , TMR2 LSB */ 
    return (timer);
}

/********************************************************************
*    Function Name:  WriteTimer1                                    *
*    Description:    This routine writes a 16-bit value to Timer0   *
*    Parameters:     unsigned int: value to write to Timer          *
*    Return Value:   None                                           *
********************************************************************/
void WriteTimer1(unsigned int timer)
{
    TMR1 = timer;     /* assign timer value to Timer1 Register */
}
/********************************************************************
*    Function Name:  WriteTimer2                                    *
*    Description:    This routine writes a 16-bit value to Timer2   *
*    Parameters:     unsigned int: value to write to Timer          *
*    Return Value:   None                                           *
********************************************************************/
void WriteTimer2(unsigned int timer)
{
    TMR2 = timer;     /* assign timer value to Timer2 Register */
}

/********************************************************************
*    Function Name:  WriteTimer3                                    *
*    Description:    This routine writes a 16-bit value to Timer3   *
*    Parameters:     unsigned int: value to write to Timer          *
*    Return Value:   None                                           *
********************************************************************/
void WriteTimer3(unsigned int timer)
{
    TMR3 = timer;     /* assign timer value to Timer3 Register */
}

/********************************************************************
*    Function Name:  WriteTimer23                                   *
*    Description:    This routine writes a LS word                  *
                     to TMR2 and MS word into TMR3                  *
*    Return Value:   None                                           *
*    Parameters:     unsigned int: value to write to TMR2 and TMR3  *
********************************************************************/
void WriteTimer23( unsigned long timer)
{
    TMR3HLD = (timer>>16);
    TMR2 = timer;       /* LSB to Timer2 and MSB to Timer3HLD */    
     
}

/* These devices support Timer 4 */
#if defined(__dsPIC30F3010__) || defined(__dsPIC30F4012__) || defined(__dsPIC30F3011__)  || \
    defined(__dsPIC30F4011__) || defined(__dsPIC30F6010__) || defined(__dsPIC30F5011__) || \
    defined(__dsPIC30F6011__) || defined(__dsPIC30F6012__) || defined(__dsPIC30F5013__) || \
    defined(__dsPIC30F6013__) || defined(__dsPIC30F6014__) || defined(__dsPIC30F4013__) || \
    defined(__dsPIC30F5015__)

/********************************************************************
*    Function Name:  CloseTimer4                                    *
*    Description:    This routine disables the Timer4 and its       *
*                    interrupt and flag bits.                       *
*    Parameters:     None                                           *
*    Return Value:   None                                           *
********************************************************************/
void CloseTimer4(void)
{
    IEC1bits.T4IE = 0;  /* Disable the Timer4 interrupt */
    T4CONbits.TON=0;    /* disable Timer4 */
    IFS1bits.T4IF = 0;  /* Clear Timer4 interrupt flag */
}

/********************************************************************
*    Function Name:  CloseTimer5                                    *
*    Description:    This routine disables the Timer5 and its       *
*                    interrupt and flag bits.                       *
*    Parameters:     None                                           *
*    Return Value:   None                                           *
********************************************************************/
void CloseTimer5(void)
{
    IEC1bits.T5IE = 0;  /* Disable the Timer5 interrupt */
    T5CONbits.TON=0;    /* disable Timer 5 */
    IFS1bits.T5IF = 0;  /* Clear Timer5 interrupt flag */
}

/********************************************************************
*    Function Name:  CloseTimer45                                   *
*    Description:    This routine disables the Timer4 and 5 and its *
*                    interrupt and flag bits                        *
*    Parameters:     None                                           *
*    Return Value:   None                                           *
********************************************************************/
void CloseTimer45(void)
{
    IEC1bits.T5IE = 0;      /* Disable the Timer5 interrupt */
    if (T4CONbits.T32 == 1)
        T4CONbits.TON = 0;  /* Disable Timer 4 */
    IFS1bits.T5IF = 0;      /* Clear Timer5 interrupt flag */
}

/*******************************************************************
*    Function Name: ConfigIntTimer4                                *
*    Description:   This Function Configures Interrupt and sets    *
*                   Interrupt Priority                             *
*    Parameters:    unsigned int config                            *
*    Return Value:  None                                           *
*******************************************************************/
void ConfigIntTimer4(unsigned int config)
{
    IFS1bits.T4IF = 0;                   /* clear IF bit */
    IPC5bits.T4IP = (config &0x0007);    /* assigning Interrupt Priority */
    IEC1bits.T4IE = (config &0x0008)>>3; /* Interrupt Enable /Disable */
}

/*******************************************************************
*    Function Name: ConfigIntTimer5                                *
*    Description:    This Function Configures Interrupt and sets   *
*                    Interrupt Priority                            *
*    Parameters:    unsigned int config                            *
*    Return Value:  None                                           *
*******************************************************************/
void ConfigIntTimer5(unsigned int config)
{
    IEC1bits.T5IE = 0;                   /* clear IF bit */
    IPC5bits.T5IP = (config &0x0007);    /* assigning Interrupt Priority */
    IEC1bits.T5IE = (config &0x0008)>>3; /* Interrupt Enable /Disable */
}
/*******************************************************************
*    Function Name:  ConfigIntTimer45                              *
*    Description:    This Function Configures Interrupt and sets   *
*                    Interrupt Priority                            *
*    Parameters:     unsigned int config                           *               
*    Return Value:   None                                          *
*******************************************************************/
void ConfigIntTimer45(unsigned int config)
{
    IFS1bits.T5IF = 0;                   /* clear IF bit */
    IPC5bits.T5IP = (config &0x0007);    /* assigning Interrupt Priority */
    IEC1bits.T5IE = (config &0x0008)>>3; /* Interrupt Enable /Disable */
}

/********************************************************************
*    Function Name:  OpenTimer4                                     *
*    Description:    This routine configures the timer control regis*
*                    ter and timer period register.                 *
*    Parameters:     config: bit definitions to configure Timer4    *
*                    period: value to be loaded to PR reg           *
*    Return Value:   None                                           *
********************************************************************/
void OpenTimer4(unsigned int config,unsigned int period)
{
    TMR4  = 0;          /* Reset Timer4 to 0x0000 */
    PR4   = period;     /* assigning Period to Timer period register */
    T4CON = config;     /* configure timer control reg */
    T4CONbits.T32 = 0;
}

/********************************************************************
*    Function Name:  OpenTimer5                                     *
*    Description:    This routine configures the timer control regis*
*                    ter and timer period register.                 *
*    Parameters:     config: bit definitions to configure Timer5    *
*                    period: value to be loaded to PR reg           *
*    Return Value:   None                                           *
********************************************************************/
void OpenTimer5(unsigned int  config,unsigned int period)
{
    TMR5  = 0;          /* Reset Timer5 to 0x0000 */
    PR5   = period;     /* Assigning Period to Timer period register.*/
    T5CON =  config;    /* configure timer control reg */
}

/********************************************************************
*    Function Name:  OpenTimer45                                    *
*    Description:    This routine configures the timer control regis*
*                    ter and timer period register.                 *
*    Parameters:     config: bit definitions to configure Timer4    *
*                    and timer5 period to timer period register     *
*    Return Value:   None                                           *
********************************************************************/
void OpenTimer45(unsigned int  config,unsigned long period)
{
    TMR4 = 0;               /* Reset Timer4 to 0x0000 */
    TMR5 = 0;               /* Reset Timer5 to 0x0000 */
    PR4 = period;           /* assigning Period to Timer period register */
    PR5 = period>>16;       /* Period to PR4 and PR5 Register */
    T4CON = config;         /* configure timer control reg */
    T4CONbits.T32 = 1;
}

/********************************************************************
*    Function Name:  ReadTimer4                                     *
*    Description:    This routine reads the 16-bit value from       *
*                    Timer4 Register.                               *
*    Parameters:     None                                           *
*    Return Value:   unsigned int: Timer  16-bit value              *
********************************************************************/
unsigned int ReadTimer4(void)
{
    return (TMR4);      /* Return the Timer4 register */
}

/********************************************************************
*    Function Name:  ReadTimer5                                     *
*    Description:    This routine reads the 16-bit value from       *
*                    Timer5 Register.                               *
*    Parameters:     None                                           *
*    Return Value:   unsigned int: Timer  16-bit value              *
********************************************************************/
unsigned int ReadTimer5(void)
{
    return (TMR5);      /* Return the Timer5 register */
}

/********************************************************************
*    Function Name:  ReadTimer45                                    *
*    Description:    This routine reads the 32-bit value from       *
*                    Timer4 and Timer5.                             *
*    Parameters:     None                                           *
*    Return Value:   unsigned long: Timer4 and Timer5  32-bit value *
********************************************************************/
unsigned long ReadTimer45(void)
{
    unsigned long timer;
    timer = TMR4;                        /* Copy Timer4 into timer low 16bit */
    timer |= (unsigned long)TMR5HLD<<16; /* shift  16 time as TMR 5 
                                            contains MSB , TMR4 LSB */ 
    return (timer);
}

/********************************************************************
*    Function Name:  WriteTimer4                                    *
*    Description:    This routine writes a 16-bit value to Timer4   *
*    Parameters:     unsigned int: value to write to Timer          *
*    Return Value:   None                                           *
********************************************************************/
void WriteTimer4(unsigned int timer)
{
    TMR4 = timer;     /* assign timer value to Timer4 Register */
}
/********************************************************************
*    Function Name:  WriteTimer5                                    *
*    Description:    This routine writes a 16-bit value to Timer5   *
*    Parameters:     int: value to write to Timer                   *
*    Return Value:   None                                           *
********************************************************************/
void WriteTimer5(unsigned int timer)
{
    TMR5 = timer;     /* assign timer value to Timer5 Register */
}

/********************************************************************
*    Function Name:  WriteTimer45                                   *
*    Description:    This routine writes a least 16-bit value       *
                     to Timer4 and Msb into Timer5                  *
*    Parameters:     unsigned long				    *
*    Return Value:   None                                           *
********************************************************************/
void WriteTimer45( unsigned long timer)
{   
    TMR5HLD = (timer>>16);
    TMR4 = timer;       /* LSB to Timer4 and MSB to Timer5HLD */          
}

#endif 

